﻿/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
using DataAccess;
using Ext.Net;
using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Text.RegularExpressions;
using System.Web;
using System.Web.UI;
using System.Web.UI.HtmlControls;
using System.Web.UI.WebControls;

//setup an alias for all direct method calls on this page
[DirectMethodProxyID(IDMode = DirectMethodProxyIDMode.Alias, Alias = "PATDEMOGRAPHICS")]

public partial class pat_demographics : System.Web.UI.Page
{
    /// <summary>
    /// Initialize controls in the page
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void Page_Init(object sender, EventArgs e)
    {
        //Enable Master Save button
        Master.EnableMasterSaveButton(true, "App.direct.PATDEMOGRAPHICS.UpdatePatientDemographics();");

        LoadStaticData();

        //pass on BaseMaster to Patient Account user control
        ucPatPortalAccount.BaseMstr = Master;
    }

    /// <summary>
    /// US:5959,5961,6967
    /// Pulls and display patient demographics from central database
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void Page_Load(object sender, EventArgs e)
    {
        if (!IsPostBack)
        {
            if (!String.IsNullOrEmpty(Master.SelectedPatientID))
            {

                //load patient demographics data
                LoadPatientDemographics();

                //load patient socioeconomics data
                LoadPatientSocioeconomics();

                //initialize the patient portal account user control
                ucPatPortalAccount.InitializeControl();
            }
        }

        pnlAccount.UpdateLayout();
    }

    /// <summary>
    /// Text to be displayed in the "null" combo selection
    /// </summary>
    const string cnNullComboOptionText = "-- Select --";

    /// <summary>
    /// US:7242 Load static data to populate dropdowns and other controls for the user to select values
    /// </summary>
    protected void LoadStaticData()
    {
        CStat stat = new CStat(Master);

        // NOTE: Need to research this, but for the checkbokgroup 
        // to return values to the code-behind in this page 
        // it has to be outside the !X.IsAjaxRequest conditional

        //Load Race
        DataSet dsRace = stat.GetDataCollectionDS("_RACE");
        CExtUtils.LoadExtCheckboxGroup(chkgrpRace, dsRace, "COLLECTION_ITEM_ID", "ITEM");
        HtmlDecodeCheckboxGroup(chkgrpRace);

        if (!X.IsAjaxRequest)
        {
            //Load States
            DataSet dsStates = stat.GetDataCollectionDS("_STATES");
            CExtUtils.LoadEXTCombo(cboState, dsStates, "COLLECTION_ITEM_ID", "ITEM");
            cboState.InsertItem(0, cnNullComboOptionText, null);
            HtmlDecodeCombo(cboState);

            //Load Gender
            DataSet dsGender = stat.GetDataCollectionDS("_GENDER");
            CExtUtils.LoadEXTCombo(cboGender, dsGender, "PLACEHOLDER2", "ITEM");
            cboGender.InsertItem(0, cnNullComboOptionText, null);
            HtmlDecodeCombo(cboGender);

            //Load Duty Status
            DataSet dsDutyStatus = stat.GetDataCollectionDS("_DUTY_STATUS");
            CExtUtils.LoadEXTCombo(cboDutyStatus, dsDutyStatus, "COLLECTION_ITEM_ID", "ITEM");
            cboDutyStatus.InsertItem(0, cnNullComboOptionText, null);
            HtmlDecodeCombo(cboDutyStatus);

            //Load Preferred Language
            DataSet dsPrefLang = stat.GetDataCollectionDS("_PREFERRED_LANGUAGE");
            CExtUtils.LoadEXTCombo(cboPreferredLanguage, dsPrefLang, "COLLECTION_ITEM_ID", "ITEM");
            cboPreferredLanguage.InsertItem(0, cnNullComboOptionText, null);
            HtmlDecodeCombo(cboPreferredLanguage);

            //Load Ethnicity
            DataSet dsEthnicity = stat.GetDataCollectionDS("_ETHNICITY");
            CExtUtils.LoadEXTCombo(cboEthnicity, dsEthnicity, "COLLECTION_ITEM_ID", "ITEM");
            cboEthnicity.InsertItem(0, cnNullComboOptionText, null);
            HtmlDecodeCombo(cboEthnicity);

            //Load Hand Preference
            DataSet dsHandPreference = stat.GetDataCollectionDS("_HAND_PREFERENCE");
            CExtUtils.LoadEXTCombo(cboHandPreference, dsHandPreference, "COLLECTION_ITEM_ID", "ITEM");
            cboHandPreference.InsertItem(0, cnNullComboOptionText, null);
            HtmlDecodeCombo(cboHandPreference);

            //Load Highest Grade
            DataSet dsHighestGrade = stat.GetDataCollectionDS("_HIGHEST_GRADE");
            CExtUtils.LoadEXTCombo(cboHighestGrade, dsHighestGrade, "COLLECTION_ITEM_ID", "ITEM");
            cboHighestGrade.InsertItem(0, cnNullComboOptionText, null);
            HtmlDecodeCombo(cboHighestGrade);

            //Load Highest Degree
            DataSet dsHighestDegree = stat.GetDataCollectionDS("_HIGHEST_DEGREE");
            CExtUtils.LoadEXTCombo(cboHighestDegree, dsHighestDegree, "COLLECTION_ITEM_ID", "ITEM");
            cboHighestDegree.InsertItem(0, cnNullComboOptionText, null);
            HtmlDecodeCombo(cboHighestDegree);

            //Load Marital Status
            DataSet dsMaritalStatus = stat.GetDataCollectionDS("_MARITAL_STATUS");
            CExtUtils.LoadEXTCombo(cboMaritalStatus, dsMaritalStatus, "COLLECTION_ITEM_ID", "ITEM");
            cboMaritalStatus.InsertItem(0, cnNullComboOptionText, null);
            HtmlDecodeCombo(cboMaritalStatus);

            //Load Employment Status
            DataSet dsEmploymentStatus = stat.GetDataCollectionDS("_EMPLOYMENT_STATUS");
            CExtUtils.LoadEXTCombo(cboEmploymentStatus, dsEmploymentStatus, "COLLECTION_ITEM_ID", "ITEM");
            cboEmploymentStatus.InsertItem(0, cnNullComboOptionText, null);
            HtmlDecodeCombo(cboEmploymentStatus);

            //Load Job Classification Category
            DataSet dsJobClassification = stat.GetDataCollectionDS("_JOB_CLASSIFICATION");
            CExtUtils.LoadEXTCombo(cboJobClassification, dsJobClassification, "COLLECTION_ITEM_ID", "ITEM");
            cboJobClassification.InsertItem(0, cnNullComboOptionText, null);
            HtmlDecodeCombo(cboJobClassification);
        }
    }

    /// <summary>
    /// US:5959
    /// Loads demographics tab with patient's demographic data from central database
    /// </summary>
    protected void LoadPatientDemographics()
    {
        CPatientDemographics patdemo = new CPatientDemographics(Master);

        DataSet dsPatDemographics = patdemo.GetPatientDemographicsDS(Master.SelectedPatientID);

        if (dsPatDemographics != null)
        {

            foreach (DataTable dt in dsPatDemographics.Tables)
            {
                foreach (DataRow dr in dt.Rows)
                {
                    //first name
                    if (!dr.IsNull("FIRST_NAME"))
                    {
                        txtFirstName.Value = dr["FIRST_NAME"].ToString();
                    }

                    //mi
                    if (!dr.IsNull("MI"))
                    {
                        txtMI.Value = dr["MI"].ToString();
                    }

                    //last name
                    if (!dr.IsNull("LAST_NAME"))
                    {
                        txtLastName.Value = dr["LAST_NAME"].ToString();
                    }

                    //ssn
                    if (!dr.IsNull("SSN"))
                    {
                        txtSSN.Value = dr["SSN"].ToString();
                        txtSSNConfirm.Value = dr["SSN"].ToString();
                    }

                    //address
                    if (!dr.IsNull("ADDRESS1"))
                    {
                        txtAddress1.Value = dr["ADDRESS1"].ToString();
                    }

                    if (!dr.IsNull("ADDRESS2"))
                    {
                        txtAddress2.Value = dr["ADDRESS2"].ToString();
                    }

                    //city
                    if (!dr.IsNull("CITY"))
                    {
                        txtCity.Value = dr["CITY"].ToString();
                    }

                    //state
                    if (!dr.IsNull("STATE_ID"))
                    {
                        SetComboBoxValue(cboState, dr["STATE_ID"].ToString());
                    }

                    //postal code
                    if (!dr.IsNull("POSTAL_CODE"))
                    {
                        txtPostalCode.Value = dr["POSTAL_CODE"].ToString();
                    }

                    //work phone
                    if (!dr.IsNull("WORKPHONE"))
                    {
                        txtWorkPhone.Value = dr["WORKPHONE"].ToString();
                    }

                    //cell/home phone
                    if (!dr.IsNull("CELLPHONE"))
                    {
                        txtCellPhone.Value = dr["CELLPHONE"].ToString();
                    }

                    //email
                    if (!dr.IsNull("EMAIL"))
                    {
                        txtEmail.Value = dr["EMAIL"].ToString();
                    }

                    //ok to call
                    if (!dr.IsNull("WRK_PHONE_CALL"))
                    {
                        chkOkCall.Checked = Convert.ToBoolean(dr["WRK_PHONE_CALL"]);
                    }

                    //ok to text
                    if (!dr.IsNull("HOME_PHONE_MSG"))
                    {
                        chkOkText.Checked = Convert.ToBoolean(dr["HOME_PHONE_MSG"]);
                    }

                    //ok to email
                    if (!dr.IsNull("EMAIL_MSG"))
                    {
                        chkOkEmail.Checked = Convert.ToBoolean(dr["EMAIL_MSG"]);
                    }

                    //gender
                    if (!dr.IsNull("GENDER"))
                    {
                        SetComboBoxValue(cboGender, dr["GENDER"].ToString());
                    }

                    //dob
                    if (!dr.IsNull("DOB"))
                    {
                        txtDOB.Value = dr["DOB"].ToString();
                    }

                    //ethnicity
                    SetComboBoxValue(cboEthnicity, GetPatientEthnicity());

                    //race
                    SetCheckboxGroupValues(chkgrpRace, GetPatientRaceList());

                    //language preference
                    if (!dr.IsNull("PREFERRED_LANGUAGE_ID"))
                    {
                        SetComboBoxValue(cboPreferredLanguage, dr["PREFERRED_LANGUAGE_ID"].ToString());
                        if (dr["PREFERRED_LANGUAGE_ID"].ToString() == "7" /*Other, specify*/)
                        {
                            txtPreferredLanguage.Text = dr["PREFERRED_LANGUAGE_OTHER"].ToString();
                            txtPreferredLanguage.Hidden = false;
                        }
                    }

                    //duty status
                    if (!dr.IsNull("DUTY_STATUS_ID"))
                    {
                        SetComboBoxValue(cboDutyStatus, dr["DUTY_STATUS_ID"].ToString());
                    }

                    //hand preference type
                    if (!dr.IsNull("HAND_PREFERENCE_ID"))
                    {
                        SetComboBoxValue(cboHandPreference, dr["HAND_PREFERENCE_ID"].ToString());
                    }
                }
            }
        }
    }

    /// <summary>
    /// US:5961 US:5960
    /// Loads socioeconomics tab with patient's socioeconomics data from central database
    /// </summary>
    protected void LoadPatientSocioeconomics()
    {
        CPatientDemographics patdemo = new CPatientDemographics(Master);

        DataSet dsSocioEcon = patdemo.GetPatientSocioeconomicsDS(Master.SelectedPatientID);

        if (dsSocioEcon != null)
        {
            foreach (DataTable dt in dsSocioEcon.Tables)
            {
                foreach (DataRow dr in dt.Rows)
                {

                    //highest grade id
                    if (!dr.IsNull("HIGHEST_GRADE_ID"))
                    {
                        SetComboBoxValue(cboHighestGrade, dr["HIGHEST_GRADE_ID"].ToString());
                    }

                    //highest degre id
                    if (!dr.IsNull("HIGHEST_DEGREE_ID"))
                    {
                        SetComboBoxValue(cboHighestDegree, dr["HIGHEST_DEGREE_ID"].ToString());
                    }

                    //marital status
                    if (!dr.IsNull("MARITAL_STATUS_ID"))
                    {
                        SetComboBoxValue(cboMaritalStatus, dr["MARITAL_STATUS_ID"].ToString());
                    }

                    //employment status
                    if (!dr.IsNull("EMPLOYMENT_STATUS_ID"))
                    {
                        SetComboBoxValue(cboEmploymentStatus, dr["EMPLOYMENT_STATUS_ID"].ToString());
                    }

                    //job classification
                    if (!dr.IsNull("JOB_CLASSIFICATION_ID"))
                    {
                        SetComboBoxValue(cboJobClassification, dr["JOB_CLASSIFICATION_ID"].ToString());
                    }
                }
            }
        }
    }

    /// <summary>
    /// Builds and returns a comma separated string with the patient's race selection values
    /// </summary>
    /// <returns></returns>
    protected string GetPatientRaceList()
    {
        string strRace = String.Empty;
        CPatientDemographics patdemo = new CPatientDemographics(Master);
        DataSet dsRace = patdemo.GetPatientRaceDS(Master.SelectedPatientID);
        if (dsRace != null)
        {
            foreach (DataTable dt in dsRace.Tables)
            {
                foreach (DataRow dr in dt.Rows)
                {
                    if (!dr.IsNull("RACE_ID"))
                    {
                        strRace += dr["RACE_ID"].ToString() + ",";
                    }
                }
            }
            if (strRace.Length > 0)
            {
                strRace = strRace.Substring(0, strRace.Length - 1);
            }
        }
        return strRace;
    }

    /// <summary>
    /// Builds and returns a string with the selected patient's ethnicity value
    /// </summary>
    /// <returns></returns>
    protected string GetPatientEthnicity()
    {
        string strEthnicity = String.Empty;
        CPatientDemographics patdemo = new CPatientDemographics(Master);
        DataSet dsEthnicity = patdemo.GetPatientEthnicityDS(Master.SelectedPatientID);
        if (dsEthnicity != null)
        {
            foreach (DataTable dt in dsEthnicity.Tables)
            {
                foreach (DataRow dr in dt.Rows)
                {
                    if (!dr.IsNull("ETHNICITY_ID"))
                    {
                        strEthnicity = dr["ETHNICITY_ID"].ToString();
                    }
                }
            }
        }
        return strEthnicity;
    }

    /// <summary>
    /// Helper method for selecting an Ext.NET combobox item with the passed on value parameter 
    /// </summary>
    /// <param name="cbo"></param>
    /// <param name="strValue"></param>
    protected void SetComboBoxValue(Ext.Net.ComboBox cbo, string strValue)
    {
        if (!String.IsNullOrEmpty(strValue))
        {
            foreach (Ext.Net.ListItem li in cbo.Items) {
                if (li.Value.Equals(strValue)) {
                    cbo.Select(li.Value);
                }
            }
        }
        else
        {
            cbo.ClearValue();
        }
    }

    /// <summary>
    /// Helper method for selecting an Ext.NET checkbox group item with the passed on value parameter 
    /// </summary>
    /// <param name="chk"></param>
    /// <param name="strValues"></param>
    protected void SetCheckboxGroupValues(Ext.Net.CheckboxGroup chk, string strValues)
    {
        // first clear all checkboxes
        foreach (Ext.Net.Checkbox c in chk.Items)
        {
            c.Checked = false;
        }

        //set values from comma separated string
        foreach (string v in strValues.Split(','))
        {
            foreach (Ext.Net.Checkbox c in chk.Items)
            {
                if (v.Equals(c.InputValue))
                {
                    c.Checked = true;
                }
            }
        }
    }

    /// <summary>
    /// Helper method to decode html characters in comboboxes options' values and texts 
    /// </summary>
    /// <param name="cbo"></param>
    protected void HtmlDecodeCombo(Ext.Net.ComboBox cbo)
    {
        foreach (Ext.Net.ListItem li in cbo.Items)
        {
            li.Text = HttpUtility.HtmlDecode(li.Text);
            li.Value = HttpUtility.HtmlDecode(li.Value);
        }
    }

    /// <summary>
    /// Helper method to decode html characters in checkbox group items' values and texts 
    /// </summary>
    /// <param name="chkg"></param>
    protected void HtmlDecodeCheckboxGroup(Ext.Net.CheckboxGroup chkg)
    {
        foreach (Ext.Net.Checkbox cb in chkg.Items)
        {
            cb.BoxLabel = Server.HtmlDecode(cb.BoxLabel);
            cb.InputValue = Server.HtmlDecode(cb.InputValue);
        }
    }

    [Ext.Net.DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// US:5912 US:5961 US:7242
    /// Collects the page data and submits the calls to the stored procedures 
    /// that update patient demographics and socioeconomics 
    /// </summary>
    /// <returns></returns>
    public void UpdatePatientDemographics()
    {
        CPatientDemographics patdemo = new CPatientDemographics(Master);
        string strValidationErrors = String.Empty;
        bool bUpdated = false;

        if (ValidatePatientDemographics(out strValidationErrors))
        {
            //update patient demographics

            //initialize variables for demographics values
            string strDOB = Convert.ToDateTime(txtDOB.Text).ToShortDateString();
            string strSSN = Regex.Replace(txtSSN.Text, "\\D", String.Empty);
            long lDutyStatusID = 0;
            long lPrefLangID = 0;
            string strPrefLang = String.Empty;
            long lHandPrefID = 0;

            //get values from the page controls and assign them to the variables 
            if (cboDutyStatus.SelectedItems.Count > 0)
            {
                lDutyStatusID = Convert.ToInt32(cboDutyStatus.SelectedItem.Value);
            }

            if (cboPreferredLanguage.SelectedItems.Count > 0)
            {
                lPrefLangID = Convert.ToInt32(cboPreferredLanguage.SelectedItem.Value);
                if (lPrefLangID == 7 /*Other, specidy*/)
                {
                    if (!String.IsNullOrEmpty(txtPreferredLanguage.Text.Trim()))
                    {
                        strPrefLang = txtPreferredLanguage.Text;
                    }
                }
                else
                {
                    //always keep the language name even when we known the value
                    strPrefLang = txtPreferredLanguage.Text;
                }
            }

            if (cboHandPreference.SelectedItems.Count > 0)
            {
                lHandPrefID = Convert.ToInt32(cboHandPreference.SelectedItem.Value);
            }

            //update demographics in DB
            bUpdated = patdemo.UpdatePatientDemographics(txtFirstName.Text,
                                            txtMI.Text,
                                            txtLastName.Text,
                                            strSSN,
                                            strSSN,
                                            cboGender.SelectedItem.Value,
                                            strDOB,
                                            String.Empty, //provider_id
                                            txtAddress1.Text,
                                            txtAddress2.Text,
                                            txtCity.Text,
                                            txtPostalCode.Text,
                                            String.Empty,
                                            txtCellPhone.Text,
                                            txtWorkPhone.Text,
                                            txtEmail.Text,
                                            cboState.SelectedItem.Value,
                                            lDutyStatusID,
                                            lPrefLangID,
                                            strPrefLang,
                                            lHandPrefID,
                                            chkOkCall.Checked ? 1 : 0,
                                            chkOkText.Checked ? 1 : 0,
                                            chkOkEmail.Checked ? 1 : 0);

            //update patient race
            UpdatePatientRace();

            //update patient ethnicity
            UpdatePatientEthnicity();

            //update patient socioeconomics

            //initialize variables for socioeconomics values
            long lHighestGradeID = 0;
            long lHighestDegreeID = 0;
            long lMaritalStatusID = 0;
            long lEmploymentStatusID = 0;
            long lJobClassificationID = 0;
            string strEmploymentStatusOther = String.Empty;

            //get values from the page controls and assign them to the variables 
            if (cboHighestGrade.SelectedItems.Count > 0)
            {
                lHighestGradeID = Convert.ToInt32(cboHighestGrade.SelectedItem.Value);
            }

            if (cboHighestDegree.SelectedItems.Count > 0)
            {
                lHighestDegreeID = Convert.ToInt32(cboHighestDegree.SelectedItem.Value);
            }

            if (cboMaritalStatus.SelectedItems.Count > 0)
            {
                lMaritalStatusID = Convert.ToInt32(cboMaritalStatus.SelectedItem.Value);
            }

            if (cboEmploymentStatus.SelectedItems.Count > 0)
            {
                lEmploymentStatusID = Convert.ToInt32(cboEmploymentStatus.SelectedItem.Value);
            }

            if (cboJobClassification.SelectedItems.Count > 0)
            {
                lJobClassificationID = Convert.ToInt32(cboJobClassification.SelectedItem.Value);
            }

            //update socioeconomics in DB
            bUpdated = bUpdated && patdemo.UpdatePatientSocioeconomics(lHighestGradeID,
                                                lHighestDegreeID,
                                                lMaritalStatusID,
                                                lEmploymentStatusID,
                                                lJobClassificationID,
                                                strEmploymentStatusOther);

            //display update confirmation
            if (bUpdated)
            {
                Notification.Show(new NotificationConfig
                {
                    PinEvent = "none",
                    Plain = true,
                    Header = false,
                    Html = "Patient data was successfully updated!"
                });
            }
            else
            {
                Master.ShowSystemFeedback("There was an error updating patient's data. Please try again. If the problem persits, contact your systems administrator.", "System Error");
            }
        }
        else {
            Master.ShowSystemFeedback(strValidationErrors, "Data Validation Errors");
        }
    }

    /// <summary>
    /// Submits patient's race selected values to the stored procedures that updates the race table 
    /// </summary>
    protected void UpdatePatientRace() {
        string strValues = String.Empty;
        foreach (Checkbox cb in chkgrpRace.CheckedItems) {
            strValues += cb.InputValue + ",";
        }

        if (!String.IsNullOrEmpty(strValues)) {
            CPatientDemographics patdemo = new CPatientDemographics(Master);
            patdemo.UpdatePatientRace(strValues);
        }
    }

    /// <summary>
    /// Submits patient's ethnicity selected value to the stored procedures that updates the race table 
    /// </summary>
    protected void UpdatePatientEthnicity() {
        long lPatientEthnicityID = 0;
        if(cboEthnicity.SelectedItems.Count > 0){
            lPatientEthnicityID = Convert.ToInt32(cboEthnicity.SelectedItem.Value);
            CPatientDemographics patdemo = new CPatientDemographics(Master);
            patdemo.UpdatePatientEthnicity(lPatientEthnicityID);
        }
    }

 /* +-------------------------------------------+
  * |                                           |
  * |         FORM DATA VALIDATIONS             |
  * |                                           |
  * +-------------------------------------------+
 */
    #region Validations

    /// <summary>
    /// Validates the Demographics and Socioeconomic submited data
    /// </summary>
    /// <returns></returns>
    protected bool ValidatePatientDemographics(out string strValidationFeedback)
    {

        bool bValidate = true;

        string strDemographicsErrors = String.Empty;
        string strSocioeconomicErrors = String.Empty;

        bValidate = ValidateDemographicsData(out strDemographicsErrors) && ValidateSocioeconomicData(out strSocioeconomicErrors);

        strValidationFeedback = strDemographicsErrors + strSocioeconomicErrors;

        return bValidate;
    }

    /// <summary>
    /// US:7242 Validates the data entered in the demographics tab
    /// </summary>
    /// <param name="strMessage"></param>
    /// <returns></returns>
    protected bool ValidateDemographicsData(out string strMessage)
    {

        //initialize the return variables
        bool bValidate = true;
        strMessage = String.Empty;
        List<string> lstErrors = new List<string>();

        // ###############################################
        // check for required fields

        //first name
        if (String.IsNullOrEmpty(txtFirstName.Text))
        {
            lstErrors.Add("Please enter the FIRST NAME.");
        }

        //last name
        if (String.IsNullOrEmpty(txtLastName.Text))
        {
            lstErrors.Add("Please enter the LAST NAME.");
        }

        //ssn
        if (String.IsNullOrEmpty(txtSSN.Text))
        {
            lstErrors.Add("Please enter the SSN.");
        }

        //dob
        if (String.IsNullOrEmpty(txtDOB.Text))
        {
            lstErrors.Add("Please enter the DATE OF BIRTH.");
        }

        // ###############################################
        // Confirm SSN
        if (!txtSSN.Text.Equals(txtSSNConfirm.Text))
        {
            lstErrors.Add("SSN and SSN Confirmation do not match.");
        }

        // ###############################################
        // validate EMAIL format

        if (txtEmail.Text.Trim().Length > 0)
        {
            if (!IsValidEmail(txtEmail.Text.Trim()))
            {
                lstErrors.Add("Please enter a valid EMAIL.");
            }
        }

        // ###############################################
        // validate DOB
        string strDOBError = String.Empty;
        if (!IsValidDOB(txtDOB.RawText.Trim(), out strDOBError))
        {
            lstErrors.Add(strDOBError);
        }

        // ###################################
        // check for GENDER

        if (cboGender.SelectedItem.Index < 1)
        {
            lstErrors.Add("Please select GENDER.");
        }

        // ###################################
        //check for PREFERRED LANGUAGE. If other, it should be specify.

        if (cboPreferredLanguage.SelectedItem.Index == 7 /*Other, specify*/)
        {
            if (String.IsNullOrEmpty(txtPreferredLanguage.Text.Trim()))
            {
                lstErrors.Add("Please specify PREFERRED LANGUAGE.");
            }
        }

        // ###################################
        // check for ETHNICITY

        if (cboEthnicity.SelectedItem.Index < 1)
        {
            lstErrors.Add("Please select ETHNICITY.");
        }

        // ###################################
        // check for RACE
        if (chkgrpRace.CheckedItems.Count < 1) {
            lstErrors.Add("Please select RACE.");
        }


        // ###############################################


        bValidate = lstErrors.Count == 0;
        if (!bValidate)
        {
            strMessage = "~#(h4)#~Demographics tab:~#(/h4)#~";
            strMessage += "~#(ul)#~";
            strMessage += TransformErrorMessage(lstErrors);
            strMessage += "~#(/ul)#~";
        }

        return bValidate;
    }

    /// <summary>
    /// Validates the data entered in the socioeconomic tab
    /// </summary>
    /// <param name="strMessage"></param>
    /// <returns></returns>
    protected bool ValidateSocioeconomicData(out string strMessage)
    {
        //TODO: Complete the validation rules
        strMessage = String.Empty;
        return true;
    }

    /// <summary>
    /// Validates an EMAIL address
    /// </summary>
    /// <param name="strEmail"></param>
    /// <returns></returns>
    public bool IsValidEmail(string strEmail)
    {
        if (String.IsNullOrEmpty(strEmail))
            return false;

        // Return true if strEmail is in valid e-mail format. 
        try
        {
            return Regex.IsMatch(strEmail,
                  @"^(?("")("".+?(?<!\\)""@)|(([0-9a-z]((\.(?!\.))|[-!#\$%&'\*\+/=\?\^`\{\}\|~\w])*)(?<=[0-9a-z])@))" +
                  @"(?(\[)(\[(\d{1,3}\.){3}\d{1,3}\])|(([0-9a-z][-\w]*[0-9a-z]*\.)+[a-z0-9][\-a-z0-9]{0,22}[a-z0-9]))$",
                  RegexOptions.IgnoreCase, TimeSpan.FromMilliseconds(500));
        }
        catch (RegexMatchTimeoutException)
        {
            return false;
        }
    }

    /// <summary>
    /// Validates the Date of Birth
    /// </summary>
    /// <param name="strDOB"></param>
    /// <returns></returns>
    protected bool IsValidDOB(string strDOB, out string strError)
    {

        bool bIsValid = true;
        strError = String.Empty;

        // ###################################
        // check if it is a valid format
        if (String.IsNullOrEmpty(strDOB))
        {
            bIsValid = false;
            strError = "Please enter a valid DOB in the format MM/DD/YYYY.";
        }

        // Return true if strDOB is in valid date format. 
        if (!Regex.IsMatch(strDOB, @"^(0[1-9]|1[012])/(0[1-9]|[12][0-9]|3[01])/(19|20)\d\d$",
                  RegexOptions.IgnoreCase))
        {
            bIsValid = false;
            strError = "Please enter a valid DOB in the format MM/DD/YYYY.";
        }

        // ###################################
        // check if DOB is not in the future
        if (bIsValid)
        {
            DateTime dtDOB;
            DateTime.TryParse(strDOB, out dtDOB);

            DateTime dtToday;
            DateTime.TryParse(DateTime.Now.ToShortDateString(), out dtToday);

            int iDateCompare = DateTime.Compare(dtDOB, dtToday);
            if (iDateCompare >= 0)
            {
                bIsValid = false;
                strError = "DOB must be a valid date in the past.";
            }
        }

        // ###################################
        // check for actual age; 
        // must be 18 yrs or older
        if (bIsValid)
        {
            DateTime dtDOB;
            DateTime.TryParse(strDOB, out dtDOB);

            DateTime dtToday = DateTime.Today;

            int iAge = dtToday.Year - dtDOB.Year;
            if (dtToday < dtDOB.AddYears(iAge))
            {
                iAge--;
            }

            if (iAge < 18)
            {
                bIsValid = false;
                strError = "Verify DOB; patient's age must be equal or greater than 18 yrs.";
            }
        }

        return bIsValid;
    }

    /// <summary>
    /// Transforms the error list (List<string>) into an HTML UL string
    /// to be displayed in the System Feedbacks window
    /// </summary>
    /// <param name="lstErrors"></param>
    /// <returns></returns>
    protected string TransformErrorMessage(List<string> lstErrors)
    {
        string[] strErrors = lstErrors.ToArray();
        string strOutput = String.Empty;

        foreach (string strE in strErrors)
        {
            strOutput += String.Format("~#(li)#~{0}~#(/li)#~", strE);
        }

        return strOutput;
    }

    #endregion
}